<?php
/*
Plugin Name: Code Embed
Description: Standalone plugin for Output.GURU that adds syntax highlighting (Visual Studio Code style) and a visible copy button to code blocks in WordPress posts.
Version: 1.0.7
Author: xAI
License: GPL2
*/

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue inline styles and scripts
 */
function code_embed_enqueue_assets() {
    // Inline CSS (Visual Studio Code Dark+ inspired)
    $css = '
        .code-embed-wrapper {
            position: relative;
            margin: 1em 0;
            overflow: visible;
        }
        .code-embed-copy-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: #252526;
            color: #d4d4d4;
            border: 1px solid #3c3c3c;
            padding: 6px 12px;
            cursor: pointer;
            font-size: 13px;
            border-radius: 4px;
            z-index: 1001;
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            box-shadow: 0 1px 3px rgba(0,0,0,0.3);
            line-height: 1;
            opacity: 1;
            transition: background 0.2s, border-color 0.2s;
        }
        .code-embed-copy-btn:hover {
            background: #333333;
            border-color: #4e4e4e;
        }
        .code-embed-copy-btn.copied {
            background: #2ea043;
            color: #ffffff;
            border-color: #2ea043;
        }
        pre {
            background: #1e1e1e;
            color: #d4d4d4;
            padding: 15px;
            border-radius: 5px;
            overflow-x: auto;
            font-family: "Consolas", "Monaco", monospace;
            position: relative;
            line-height: 1.5;
            margin: 0;
            z-index: 0;
        }
        code {
            font-family: "Consolas", "Monaco", monospace;
        }
        .code-keyword { color: #569cd6; }
        .code-string { color: #ce9178; }
        .code-comment { color: #6a9955; font-style: italic; }
        .code-number { color: #b5cea8; }
        .code-operator { color: #d4d4d4; }
        .code-tag { color: #569cd6; }
        .code-attr { color: #9cdcfe; }
        .code-selector { color: #d7ba7d; }
        .code-boolean { color: #569cd6; }
        .code-null { color: #569cd6; }
        .code-key { color: #9cdcfe; }
    ';
    wp_register_style('code-embed-inline', false);
    wp_enqueue_style('code-embed-inline');
    wp_add_inline_style('code-embed-inline', preg_replace('/\s+/', ' ', $css));

    // Inline JavaScript for copy functionality
    $js = '
        function codeEmbedCopy(button) {
            try {
                const codeElement = button.nextElementSibling.querySelector("code");
                if (!codeElement) throw new Error("Code element not found");
                const code = codeElement.innerText;
                navigator.clipboard.writeText(code).then(() => {
                    button.textContent = "Copied!";
                    button.classList.add("copied");
                    setTimeout(() => {
                        button.textContent = "Copy";
                        button.classList.remove("copied");
                    }, 2000);
                }).catch(err => {
                    console.error("Copy failed: ", err);
                    button.textContent = "Copy Failed";
                    button.classList.add("copied");
                    setTimeout(() => {
                        button.textContent = "Copy";
                        button.classList.remove("copied");
                    }, 2000);
                });
            } catch (err) {
                console.error("Copy error: ", err);
                button.textContent = "Error";
                button.classList.add("copied");
                setTimeout(() => {
                    button.textContent = "Copy";
                    button.classList.remove("copied");
                }, 2000);
            }
        }
    ';
    wp_register_script('code-embed-inline', '', [], false, true);
    wp_enqueue_script('code-embed-inline');
    wp_add_inline_script('code-embed-inline', $js);
}
add_action('wp_enqueue_scripts', 'code_embed_enqueue_assets');

/**
 * Modify code blocks in post content
 */
function code_embed_modify_content($content) {
    if (!is_singular()) {
        return $content;
    }

    // Match <pre><code> blocks with or without language class
    $pattern = '/<pre[^>]*><code(?: class="language-([a-z0-9+]+)")?>([\s\S]*?)<\/code><\/pre>/i';

    $content = preg_replace_callback($pattern, function($matches) {
        $language = isset($matches[1]) && $matches[1] ? strtolower($matches[1]) : 'plaintext';
        $code = $matches[2];

        // Highlight syntax if possible
        $code = code_embed_highlight($code, $language);

        // Wrap with copy button
        $output  = '<div class="code-embed-wrapper">';
        $output .= '<button class="code-embed-copy-btn" onclick="codeEmbedCopy(this)">Copy</button>';
        $output .= '<pre><code class="language-' . esc_attr($language) . '">' . $code . '</code></pre>';
        $output .= '</div>';

        return $output;
    }, $content);

    return $content;
}
add_filter('the_content', 'code_embed_modify_content');

/**
 * Basic syntax highlighting for supported languages
 */
function code_embed_highlight($code, $language) {
    $code = html_entity_decode($code, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $patterns = [];

    switch ($language) {
        case 'javascript':
        case 'js':
            $patterns = [
                '/\b(break|case|catch|class|const|continue|debugger|default|delete|do|else|export|extends|finally|for|function|if|import|in|instanceof|new|null|return|super|switch|this|throw|try|typeof|var|void|while|with|yield)\b/' => '<span class="code-keyword">$1</span>',
                '/("|\')(.*?)\1/' => '<span class="code-string">$1$2$1</span>',
                '/(\/\/.*$)|(\/\*[\s\S]*?\*\/)/m' => '<span class="code-comment">$1$2</span>',
                '/\b(\d+\.?\d*)\b/' => '<span class="code-number">$1</span>',
                '/(\+|-|\*|\/|%|=|==|===|!=|!==|>|<|>=|<=|&&|\|\||\?|:)/' => '<span class="code-operator">$1</span>',
            ];
            break;

        case 'php':
            $patterns = [
                '/(&lt;\?php|\?&gt;)/' => '<span class="code-tag">$1</span>',
                '/\b(abstract|and|array|as|break|case|catch|class|const|continue|declare|default|die|do|echo|else|elseif|empty|enddeclare|endfor|endforeach|endif|endswitch|endwhile|eval|exit|extends|final|finally|for|foreach|function|global|goto|if|implements|include|include_once|instanceof|insteadof|interface|isset|list|namespace|new|or|parent|print|private|protected|public|require|require_once|return|static|switch|throw|trait|try|unset|use|var|while|xor|yield)\b/' => '<span class="code-keyword">$1</span>',
                '/("|\')(.*?)\1/' => '<span class="code-string">$1$2$1</span>',
                '/(\/\/.*$)|(\/\*[\s\S]*?\*\/)|(#.*$)/m' => '<span class="code-comment">$1$2$3</span>',
                '/\b(\d+\.?\d*)\b/' => '<span class="code-number">$1</span>',
            ];
            break;

        case 'json':
            $patterns = [
                '/"([^"]+)"(?=\s*:)/' => '<span class="code-key">"$1"</span>',
                '/"([^"]*)"/' => '<span class="code-string">"$1"</span>',
                '/\b(-?\d+\.?\d*)\b/' => '<span class="code-number">$1</span>',
                '/\b(true|false)\b/' => '<span class="code-boolean">$1</span>',
                '/\b(null)\b/' => '<span class="code-null">$1</span>',
                '/([{}\[\],:])/' => '<span class="code-operator">$1</span>',
            ];
            break;

        default:
            $patterns = [
                '/("|\')(.*?)\1/' => '<span class="code-string">$1$2$1</span>',
                '/(\/\/.*$)|(\/\*[\s\S]*?\*\/)|(#.*$)/m' => '<span class="code-comment">$1$2$3</span>',
            ];
    }

    foreach ($patterns as $pattern => $replacement) {
        $code = preg_replace($pattern, $replacement, $code);
    }

    // Do NOT escape here — return highlighted HTML directly
    return $code;
}
?>
